/*  GWhere
 *  Copyright (C) 2000  Sbastien LECACHEUR
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*!
 * @header	gwdirinfo.h
 *		GWhere<br>
 *		Copyright (C) 2000  Sbastien LECACHEUR<br><br>
 *		This Distributed under the terms of the <a href="http://www.gnu.org/copyleft/gpl.html">GNU General Public Licence</a><br><br>
 *		This component was developed for the GWhere project.<br>
 *		It's an object for directory representation. His structure gets many informations as number of subfolders, total size,...<br><br>
 * @abstract	This is the directory data structure.
 * @author	Sbastien LECACHEUR <zero@gwhere.org>
 * @version	1.0 2001/08/21
 * @see		glib.h
 * @see		gwsupport.h
 * @link		GWhere Web Site <http://www.gwhere.org>
 * @link		GTK+ Web Site <http://www.gtk.org>
 * @ref		gwdirinfo.c
 * @author	Author <zero@gwhere.org>
 */


#ifndef GW_DATA_DIR_INFO_H
#define GW_DATA_DIR_INFO_H


#include <glib.h>


/*!
 * @struct	dir_info
 * @abstract	Directory information data structure
 * @discussion	Directory information is an opaque data structure containing information about directory properties and informations.
 */
struct dir_info;


/*!
 * @function	dir_info_new
 * @abstract	Creates a new directory information.
 * @discussion	This function creates a new directory information. It's an allocation function.
 * @result	a pointer to the directory information structure.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
struct dir_info * dir_info_new ( void);


/*!
 * @function	dir_info_set_sub_folder
 * @abstract	Sets the directory's number of subfolders.
 * @discussion	This function sets the number of subfolders which are in directory.<br>
 * @param	p		the directory information to set the number of subfolders
 * @param	nb		the number of subfolders
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint dir_info_set_sub_folder ( struct dir_info *p, gint nb);


/*!
 * @function	dir_info_set_size
 * @abstract	Sets the directory's size.
 * @discussion	This function sets the directory's total size.<br>
 * @param	p		the directory information to set the size
 * @param	size	the size
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint dir_info_set_size ( struct dir_info *p, guint64 size);


/*!
 * @function	dir_info_add_sub_folder
 * @abstract	Adds a subfolder to the directory.
 * @discussion	This function adds a subfolder to the directory. This function increments the number of subfolders by 1.<br>
 * @param	p		the directory information to add subfolder
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint dir_info_add_sub_folder ( struct dir_info *p);


/*!
 * @function	dir_info_add_n_sub_folder
 * @abstract	Adds a number of subfolders to the directory.
 * @discussion	This function adds a number of subfolders to the directory. This function increments the number of subfolders by n.<br>
 * @param	p		the directory information to add subfolders
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint dir_info_add_n_sub_folder ( struct dir_info *p, gint n);


/*!
 * @function	dir_info_add_size
 * @abstract	Adds a size to the directory's size.
 * @discussion	This function adds a size to the directory's total size. This function increments the number of subfolders by the size to add.<br>
 * @param	p		the directory information to add size
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint dir_info_add_size ( struct dir_info *p, guint64 size);


/*!
 * @function	dir_info_get_sub_folder
 * @abstract	Gets the directory's number of subfolders.
 * @discussion	This function gets the directory's subfolders.<br>
 * @param	p		the directory information to get the number of subfolders
 * @result	the number of subfolders.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint dir_info_get_sub_folder ( struct dir_info *p);


/*!
 * @function	dir_info_get_size
 * @abstract	Gets the directory's number of subfolders.
 * @discussion	This function gets the directory's subfolders.<br>
 * @param	p		the directory information to get the number of subfolders
 * @result	the number of subfolders.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
guint64 dir_info_get_size ( struct dir_info *p);


/*!
 * @function	dir_info_free
 * @abstract	Frees datas structure.
 * @discussion	This function frees the directory information's datas.
 * @param	p		the directory information to free
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint dir_info_free ( struct dir_info *p);


#endif
